/* Copyright (C) 2015-2018 RealVNC Ltd.  All Rights Reserved.
 */

#include <vnccommon/Log.h>

#include <vnccommon/StringUtils.h>

#include <cstdio>
#include <ostream>

#ifdef WINCE
#define vsnprintf _vsnprintf
#endif

using namespace vnccommon;

LogHandler::Tagged LogHandler::Tagged::appendTag(const std::string& tag)
{
    return Tagged(mLogHandler, mTag + "::" + tag);
}

LogHandler::Tagged LogHandler::tag(const std::string& tag)
{
    return Tagged(*this, tag);
}

void LogHandler::logV(LogLevel level, const char* tag, const char* format, va_list ap)
{
    const std::string str = StringUtils::vformat(format, ap);

    writeRaw(level, tag, str.c_str());
}

LogHandler::Tagged::Tagged(LogHandler& logHandler, const std::string& tag)
        :   mLogHandler(logHandler),
            mTag(tag)
{
}

void LogHandler::Tagged::debug(const char* format, ...)
{
    va_list ap;
    va_start(ap, format);
    mLogHandler.logV(LOG_DEBUG, mTag.c_str(), format, ap);
    va_end(ap);
}

void LogHandler::Tagged::info(const char* format, ...)
{
    va_list ap;
    va_start(ap, format);
    mLogHandler.logV(LOG_INFO, mTag.c_str(), format, ap);
    va_end(ap);
}

void LogHandler::Tagged::warning(const char* format, ...)
{
    va_list ap;
    va_start(ap, format);
    mLogHandler.logV(LOG_WARNING, mTag.c_str(), format, ap);
    va_end(ap);
}

void LogHandler::Tagged::error(const char* format, ...)
{
    va_list ap;
    va_start(ap, format);
    mLogHandler.logV(LOG_ERROR, mTag.c_str(), format, ap);
    va_end(ap);
}

LogHandler::~LogHandler()
{
}

OStreamLogHandler::OStreamLogHandler(std::ostream& stream)
        :   mStartTime(Clock::getCurrentTimeMonotonic()),
            mStream(stream)
{
}

void OStreamLogHandler::writeRaw(LogLevel level, const char* tag, const char* text)
{
    const Duration timestamp = Clock::getCurrentTimeMonotonic() - mStartTime;

    const vnc_int64_t timestamp_sec = timestamp.getMs() / 1000;
    const vnc_int64_t timestamp_msec = timestamp.getMs() % 1000;

    Mutex::Locker lock(mMutex);
    mStream << StringUtils::format(
            "[%lld.%03lld] [%s] [%d] %s",
            static_cast<long long unsigned>(timestamp_sec),
            static_cast<long long unsigned>(timestamp_msec),
            tag,
            static_cast<int>(level),
            text) << std::endl;
}

